param (
    [Parameter(Mandatory=$true)][string]$inputPath,
    [Parameter(Mandatory=$true)][string]$outputPath,
    [string]$keepActive = "false"
)

# -------------------------------
# Hilfsfunktionen
# -------------------------------

function Test-FileExists([string]$path) {
    if (-not (Test-Path -LiteralPath $path)) {
        Write-Error "Input file does not exist: $path"
        exit 1
    }
}

function Ensure-Directory([string]$dir) {
    if (-not (Test-Path -LiteralPath $dir)) {
        New-Item -ItemType Directory -Path $dir | Out-Null
    }
}

function Get-SOfficePath {
    # 1) PATH via where.exe
    try {
        $soffice = (& where.exe soffice 2>$null | Select-Object -First 1)
        if ($soffice -and (Test-Path $soffice)) { return $soffice }
    } catch {}

    # 2) Typische Installationspfade
    $candidates = @(
        "C:\Program Files\LibreOffice\program\soffice.exe",
        "C:\Program Files (x86)\LibreOffice\program\soffice.exe",
        "C:\Program Files\OpenOffice 4\program\soffice.exe",
        "C:\Program Files (x86)\OpenOffice 4\program\soffice.exe"
    )
    foreach ($c in $candidates) {
        if (Test-Path $c) { return $c }
    }
    return $null
}

function Convert-With-Word([string]$inPath, [string]$outPath, [bool]$keepOpen) {
    $word = $null
    $doc  = $null
    try {
        $word = New-Object -ComObject Word.Application
    } catch {
        throw "Word COM-Schnittstelle nicht verfügbar."
    }

    try {
        # Word-Einstellungen
        $word.Visible = $keepOpen
        $word.DisplayAlerts = 0

        # Dokument öffnen (ReadOnly, ohne Konvertierungsdialoge)
        $doc = $word.Documents.Open($inPath, $false, $true, $false)

        # Zielordner anlegen
        $outDir = [System.IO.Path]::GetDirectoryName($outPath)
        Ensure-Directory $outDir

        # 17 = wdFormatPDF
        $wdFormatPDF = 17
        $doc.SaveAs([ref]$outPath, [ref]$wdFormatPDF)
        Write-Host "Konvertierung (Word) abgeschlossen: $inPath -> $outPath"
    } finally {
        if ($doc -ne $null) { $doc.Close($false) | Out-Null }
        if (-not $keepOpen -and $word -ne $null) { $word.Quit() | Out-Null }

        # COM sauber freigeben
        if ($doc)  { [System.Runtime.Interopservices.Marshal]::ReleaseComObject($doc)  | Out-Null }
        if ($word) { [System.Runtime.Interopservices.Marshal]::ReleaseComObject($word) | Out-Null }
        [GC]::Collect()
        [GC]::WaitForPendingFinalizers()
    }
}

function Convert-With-SOffice([string]$inPath, [string]$outPath) {
    $soffice = Get-SOfficePath
    if (-not $soffice) {
        throw "Libre/OpenOffice (soffice.exe) nicht gefunden."
    }

    $targetDir  = [System.IO.Path]::GetDirectoryName($outPath)
    $targetName = [System.IO.Path]::GetFileName($outPath)
    $baseName   = [System.IO.Path]::GetFileNameWithoutExtension($targetName)

    Ensure-Directory $targetDir

    # soffice kann keinen exakten Zieldateinamen setzen, nur den Ordner.
    # Workaround: in Zielordner konvertieren und ggf. umbenennen.
    $args = @(
        "--headless",
        "--nologo",
        "--nodefault",
        "--nofirststartwizard",
        "--convert-to", "pdf:writer_pdf_Export",
        "--outdir", "`"$targetDir`"",
        "`"$inPath`""
    )

    $p = Start-Process -FilePath $soffice -ArgumentList $args -NoNewWindow -PassThru -Wait
    if ($p.ExitCode -ne 0) {
        throw "soffice-Konvertierung fehlgeschlagen (ExitCode=$($p.ExitCode))."
    }

    # Erwartete Ausgabedatei hat den Basenamen der Eingabe
    $expected = Join-Path $targetDir ("{0}.pdf" -f ([System.IO.Path]::GetFileNameWithoutExtension($inPath)))
    if (-not (Test-Path $expected)) {
        # Manche Versionen schreiben anders — fallback: suche die neueste PDF im Zielordner
        $latestPdf = Get-ChildItem -LiteralPath $targetDir -Filter *.pdf | Sort-Object LastWriteTime -Descending | Select-Object -First 1
        if ($latestPdf) { $expected = $latestPdf.FullName }
    }

    if (-not (Test-Path $expected)) {
        throw "Ausgabedatei nach Konvertierung nicht gefunden."
    }

    # Falls gewünschter Name abweicht, umbenennen
    if ($expected -ne $outPath) {
        if (Test-Path $outPath) { Remove-Item -LiteralPath $outPath -Force }
        Rename-Item -LiteralPath $expected -NewName $targetName
    }

    Write-Host "Konvertierung (Libre/OpenOffice) abgeschlossen: $inPath -> $outPath"
}

# -------------------------------
# Hauptlogik
# -------------------------------

$keepActiveBoolean = $keepActive -eq 'true'

# Eingabe prüfen
Test-FileExists -path $inputPath

# Pfade normalisieren
$inputPath  = (Resolve-Path -LiteralPath $inputPath).Path
# outputPath kann neu sein – nur Verzeichnis prüfen/erstellen
$outDir = [System.IO.Path]::GetDirectoryName($outputPath)
if ([string]::IsNullOrWhiteSpace($outDir)) {
    Write-Error "Ungültiger outputPath: $outputPath"
    exit 1
}

try {
    # 1) Versuche Word-COM
    try {
        Convert-With-Word -inPath $inputPath -outPath $outputPath -keepOpen $keepActiveBoolean
        exit 0
    } catch {
        Write-Verbose "Word-Konvertierung nicht möglich oder fehlgeschlagen: $($_.Exception.Message)"
    }

    # 2) Fallback: Libre/OpenOffice
    Convert-With-SOffice -inPath $inputPath -outPath $outputPath
    exit 0
}
catch {
    Write-Error $_.Exception.Message
    exit 1
}
